import { describe } from 'mocha';
import { expect } from 'chai';
import {
  FameContracts,
  deployContractsForTests,
  getSigners,
} from './helpers/deploy-helper';
import {
  initializeContracts,
  initializeTradingManagmentContract,
} from './helpers/contracts-initialization-helper';
import { ethers } from 'hardhat';
import { Signers } from './types';
import {
  DataAccessSubscription,
  Governance,
  OfferingToken,
  PaymentToken,
  Bourse,
  TradingManagementExecutorFacet,
} from '../../types';
import { deployDiamond } from '../../scripts/deployTradingManagement';

// The following numbers can be changed to run smaller or bigger scale test
const dp = 5; // number of data providers
const assetsPerDP = 5; // number of assets per data provider
const offersPerAsset = 2; // number of offers per asset
const totalUsers = 10; // total number of users
const purchasesPerUser = 2; // number of purchases per user
const totalAssets = dp * assetsPerDP;

let govadmin;
let paymentContract: PaymentToken;
let dataAccessSubscriptionContract: DataAccessSubscription;
let bourseContract: Bourse;
let bourseAddress;
let governanceContract: Governance;
let offeringTokenContract: OfferingToken;
let contracts: FameContracts;
let signers: Signers;
let tradingManagementExecutorFacet: TradingManagementExecutorFacet;

async function createRandomWallets(numOfWallets) {
  return Array.from({ length: numOfWallets }, () =>
    ethers.Wallet.createRandom().connect(ethers.provider),
  );
}

async function distributeEtherToWallets(wallets, senderWallet, etherAmount) {
  for (const recipient of wallets) {
    try {
      const tx = {
        to: recipient.address,
        value: ethers.utils.parseEther(etherAmount),
        gasLimit: 21000, // standard ETH transfer
      };

      const txResponse = await senderWallet.sendTransaction(tx);
    } catch (error) {
      console.error(`Failed to send to ${recipient.address}:`, error.message);
    }
  }
}

async function dataProvidersAddAssetOfferings(
  offering,
  admin,
  wallets,
  dp,
  assetsPerDP,
  assetPrefix,
) {
  for (let i = 0; i < dp; i++) {
    const dataProvider = wallets[i];
    for (let j = 0; j < assetsPerDP; j++) {
      const assetId = `${assetPrefix}-asset-${i}-${j}`;
      for (let k = 0; k < offersPerAsset; k++) {
        const oid = `${assetPrefix}-oid-${i}-${j}-${k}`;
        try {
          const tx = await offering
            .connect(admin)
            .addAsset(
              assetId,
              oid,
              'https://uri.com',
              dataProvider.address,
              100,
              1234,
              '10',
              '100MB',
              'target',
              'sl',
              '0x00',
            );
          const receipt = await tx.wait();
        } catch (e) {
          console.error('addAsset failed', e);
        }
      }
    }
  }
}

async function usersPurchaseAssetOffers(
  gov,
  tm,
  erc20,
  bourseAddress,
  admin,
  wallets,
  dp,
  assetsPerDP,
  purchaseMethod,
  assetPrefix,
) {
  const coinAmt = ethers.utils.parseEther('900');
  for (let k = 0; k < totalUsers; k++) {
    const user = wallets[dp + k];
    const usedOids = new Set<string>();
    let oid;

    try {
      await gov.connect(admin).mintCoin(user.address, 'FDE', coinAmt);
      await erc20.connect(user).approve(bourseAddress, coinAmt);
      for (let p = 0; p < purchasesPerUser; p++) {
        do {
          oid = `${assetPrefix}-oid-${Math.floor(
            Math.random() * dp,
          )}-${Math.floor(Math.random() * assetsPerDP)}-${Math.floor(
            Math.random() * offersPerAsset,
          )}`;
        } while (usedOids.has(oid));
        usedOids.add(oid);
        const data = ethers.utils.hexlify(
          ethers.utils.toUtf8Bytes(JSON.stringify({ purchasedOfferId: oid })),
        );
        const tx = await tm.connect(user)[purchaseMethod](oid, data, 10000);
      }
    } catch (e) {
      console.error('purchaseAccessRight: ', purchaseMethod, 'failed', e);
    }
  }
}

describe('97 - Users purchase offers from data providers and offerids information is queried/deleted', function () {
  beforeEach(async function () {
    contracts = await deployContractsForTests({ shouldResetNetwork: false });
    signers = await getSigners();
    await initializeContracts({ contracts, signers });
    dataAccessSubscriptionContract = contracts.dataAccessSubscriptionContract;
    bourseContract = contracts.bourseContract;
    bourseAddress = bourseContract.address;
    paymentContract = contracts.paymentTokenContract;
    governanceContract = contracts.governanceContract;
    offeringTokenContract = contracts.offeringTokenContract;
    govadmin = signers.admin;

    const tradingManagementAddress = await deployDiamond({
      saveInfo: false,
      showInfo: false,
    });

    await initializeTradingManagmentContract({
      contracts,
      signers,
      tradingManagementAddress,
    }).then((res) => {
      tradingManagementExecutorFacet = res.tradingManagementExecutorFacet;
    });
  });

  it('Retrieval/deletion test of users subscription offerids performed correctly', async function () {
    const assetPrefix = [...Array(6)]
      .map(() => Math.random().toString(36)[2])
      .join('');
    const purchaseModel = 'Subscription';
    const purchaseMethod = `purchaseAccessRight${purchaseModel}`;
    const wallets = await createRandomWallets(totalUsers + dp);
    await distributeEtherToWallets(wallets, govadmin, '0.2');
    await dataProvidersAddAssetOfferings(
      offeringTokenContract,
      govadmin,
      wallets,
      dp,
      assetsPerDP,
      assetPrefix,
    );
    await usersPurchaseAssetOffers(
      governanceContract,
      tradingManagementExecutorFacet,
      paymentContract,
      bourseAddress,
      govadmin,
      wallets,
      dp,
      assetsPerDP,
      purchaseMethod,
      assetPrefix,
    );
    for (let k = 0; k < totalUsers; k++) {
      const userAddress = wallets[dp + k].address;
      const userAllOfferids =
        await tradingManagementExecutorFacet.getAllUserDataAccessSubscriptionOfferIds(
          userAddress,
        );
      expect(userAllOfferids.length).to.equal(purchasesPerUser);
      const count =
        await tradingManagementExecutorFacet.getUserDataAccessSubscriptionOfferIdCount(
          userAddress,
        );
      expect(count.toNumber()).to.equal(purchasesPerUser);

      const offerIdToRemove = userAllOfferids[0];
      await tradingManagementExecutorFacet
        .connect(govadmin)
        .deleteUserDataAccessSubscriptionOfferId(userAddress, offerIdToRemove);
      const updatedOfferIds =
        await tradingManagementExecutorFacet.getAllUserDataAccessSubscriptionOfferIds(
          userAddress,
        );
      expect(updatedOfferIds).to.not.include(offerIdToRemove);
    }
  });

  it('Retrieval/deletion test of users PAYG offerids performed correctly', async function () {
    const assetPrefix = [...Array(6)]
      .map(() => Math.random().toString(36)[2])
      .join('');
    const purchaseModel = 'PAYG';
    const purchaseMethod = `purchaseAccessRight${purchaseModel}`;
    const wallets = await createRandomWallets(totalUsers + dp);
    await distributeEtherToWallets(wallets, govadmin, '0.1');
    await dataProvidersAddAssetOfferings(
      offeringTokenContract,
      govadmin,
      wallets,
      dp,
      assetsPerDP,
      assetPrefix,
    );
    await usersPurchaseAssetOffers(
      governanceContract,
      tradingManagementExecutorFacet,
      paymentContract,
      bourseAddress,
      govadmin,
      wallets,
      dp,
      assetsPerDP,
      purchaseMethod,
      assetPrefix,
    );
    for (let k = 0; k < totalUsers; k++) {
      const userAddress = wallets[dp + k].address;
      const userAllOfferids =
        await tradingManagementExecutorFacet.getAllUserDataAccessPAYGOfferIds(
          userAddress,
        );
      expect(userAllOfferids.length).to.equal(purchasesPerUser);
      const count =
        await tradingManagementExecutorFacet.getUserDataAccessPAYGOfferIdCount(
          userAddress,
        );
      expect(count.toNumber()).to.equal(purchasesPerUser);

      const offerIdToRemove = userAllOfferids[0];
      await tradingManagementExecutorFacet
        .connect(govadmin)
        .deleteUserDataAccessPAYGOfferId(userAddress, offerIdToRemove);
      const updatedOfferIds =
        await tradingManagementExecutorFacet.getAllUserDataAccessPAYGOfferIds(
          userAddress,
        );
      expect(updatedOfferIds).to.not.include(offerIdToRemove);
    }
  });

  it('Retrieval/deletion test of users PAYU offerids performed correctly', async function () {
    const assetPrefix = [...Array(6)]
      .map(() => Math.random().toString(36)[2])
      .join('');
    const purchaseModel = 'PAYU';
    const purchaseMethod = `purchaseAccessRight${purchaseModel}`;
    const wallets = await createRandomWallets(totalUsers + dp);
    await distributeEtherToWallets(wallets, govadmin, '0.1');
    await dataProvidersAddAssetOfferings(
      offeringTokenContract,
      govadmin,
      wallets,
      dp,
      assetsPerDP,
      assetPrefix,
    );
    await usersPurchaseAssetOffers(
      governanceContract,
      tradingManagementExecutorFacet,
      paymentContract,
      bourseAddress,
      govadmin,
      wallets,
      dp,
      assetsPerDP,
      purchaseMethod,
      assetPrefix,
    );
    for (let k = 0; k < totalUsers; k++) {
      const userAddress = wallets[dp + k].address;

      const userAllOfferids =
        await tradingManagementExecutorFacet.getAllUserDataAccessPAYUOfferIds(
          userAddress,
        );
      expect(userAllOfferids.length).to.equal(purchasesPerUser);

      const count =
        await tradingManagementExecutorFacet.getUserDataAccessPAYUOfferIdCount(
          userAddress,
        );
      expect(count.toNumber()).to.equal(purchasesPerUser);

      const offerIdToRemove = userAllOfferids[0];
      await tradingManagementExecutorFacet
        .connect(govadmin)
        .deleteUserDataAccessPAYUOfferId(userAddress, offerIdToRemove);
      const updatedOfferIds =
        await tradingManagementExecutorFacet.getAllUserDataAccessPAYUOfferIds(
          userAddress,
        );
      expect(updatedOfferIds).to.not.include(offerIdToRemove);
    }
  });

  it('Returns correct token IDs as BigNumbers', async function () {
    const assetPrefix = 'token-id-test';
    const subOid = `${assetPrefix}-oid-0-0-0`;
    const paygOid = `${assetPrefix}-oid-0-1-0`;
    const userWallet = (await createRandomWallets(1))[0];
    const dpWallets = await createRandomWallets(dp);

    await distributeEtherToWallets([userWallet, ...dpWallets], govadmin, '0.5');

    await dataProvidersAddAssetOfferings(
      offeringTokenContract,
      govadmin,
      dpWallets,
      dp,
      assetsPerDP,
      assetPrefix,
    );

    const coinAmt = ethers.utils.parseEther('900');
    await governanceContract
      .connect(govadmin)
      .mintCoin(userWallet.address, 'FDE', coinAmt);
    await paymentContract.connect(userWallet).approve(bourseAddress, coinAmt);
    const data = ethers.utils.toUtf8Bytes('');

    await tradingManagementExecutorFacet
      .connect(userWallet)
      .purchaseAccessRightSubscription(subOid, data, 10000);

    await tradingManagementExecutorFacet
      .connect(userWallet)
      .purchaseAccessRightPAYG(paygOid, data, 10000);

    const expectedSubTokenId = await offeringTokenContract.getIDHash(subOid);
    const expectedPaygTokenId = await offeringTokenContract.getIDHash(paygOid);

    // Retrieve the stored token IDs from the contract
    const returnedSubTokenIds =
      await tradingManagementExecutorFacet.getAllUserDataAccessSubscriptionOfferIds(
        userWallet.address,
      );
    const returnedPaygTokenIds =
      await tradingManagementExecutorFacet.getAllUserDataAccessPAYGOfferIds(
        userWallet.address,
      );

    // Check that the arrays contain exactly one item each
    expect(returnedSubTokenIds.length).to.equal(1);
    expect(returnedPaygTokenIds.length).to.equal(1);

    // Verify the returned value IS the correct token ID
    const returnedTokenId = returnedSubTokenIds[0];

    // Assert that the returned item is a BigNumber
    expect(ethers.BigNumber.isBigNumber(returnedTokenId)).to.be.true;

    // Assert that the returned BigNumber equals the expected BigNumber
    expect(returnedTokenId).to.equal(expectedSubTokenId);
    expect(returnedPaygTokenIds[0]).to.equal(expectedPaygTokenId);
  });
});
